#include <sys\timeb.h>
#include <stdio.h>
#include <windows.h>
#include <timeapi.h>

#include "Timing.h"
#include "Log.h"

#define TARGET_RESOLUTION_MS 1
UINT _windows_timer_min_resolution;
Uint8 _timer_resolution_set = 0;

Uint32 _timing_tstates_per_scanline__default;
Uint64 _timing_tstates_per_video_frame__default;

Uint32 _timing_tstates_per_scanline;
Uint64 _timing_tstates_per_video_frame;

Uint16 _timing_milliseconds_per_video_frame;
Uint8 _timing_tstate_factor;

double _timing_actualmilliseconds_per_video_frame;

Uint64 _idleTimeWhileCPUFaster = 0;
Uint64 _idleTimeAll = 0;

Uint8 _timing_is_ready_to_destroy = 0;

Uint32 _last_scanline_tstate_overflow;
Uint32 _per_scanline_adjustment_countdown_after_factor_change = 0;

// used to keep track of real time spent while CPU is in faster mode
struct timeb _fasterTimingStart;
Uint64 _fasterTotalRunTimeMs = 0;

// EXPERIMENTALLY DETERMINED
#define PER_SCANLINE_TSTATE_COMPENSATION_FACTOR 1.5f

#define _DEBUG_BUFFER_SIZE 256
static char _debugBuffer[_DEBUG_BUFFER_SIZE];


Uint32 timing_get_states_per_scanline() {
	// we make this scan line pay the tstates borrowed by the previous
	// a closer to 100% means less sound desynchronization
	Uint32 compensatedOverflow = (Uint32)(PER_SCANLINE_TSTATE_COMPENSATION_FACTOR * (double)_last_scanline_tstate_overflow);
	return _timing_tstates_per_scanline - compensatedOverflow;
}

void _timing_record_overflow(Uint32 overflowTstates) {
	_last_scanline_tstate_overflow = overflowTstates;
}

void _timing_reset_overflow() {
	_last_scanline_tstate_overflow = 0;
}

void _timing_handle_adjustment_after_tstate_factor_change(Uint32 actualTstates) {
	if (_per_scanline_adjustment_countdown_after_factor_change > 0) {
		// countdown after a tstate factor change
		_per_scanline_adjustment_countdown_after_factor_change--;
		_timing_reset_overflow();
		return;
	}

	if (actualTstates <= _timing_tstates_per_scanline) {
		_timing_reset_overflow();
		return;
	}

	// we had an overflow
	Uint32 overflow = actualTstates - _timing_tstates_per_scanline; // guaranteed positive (see check above)
	_timing_record_overflow(overflow);
}

void timing_report_actual_states_last_scanline(Uint32 actualTstates) {
	_timing_handle_adjustment_after_tstate_factor_change(actualTstates);
}

Uint64 timing_get_states_per_video_frame() {
	return _timing_tstates_per_video_frame;
}

Uint16 timing_get_target_milliseconds_per_video_frame() {
	return _timing_milliseconds_per_video_frame;
}

void _timing_apply_tstate_factor__worker(factor) {
	_timing_reset_overflow();
	_per_scanline_adjustment_countdown_after_factor_change = 5;

	_timing_tstate_factor = factor;
	_timing_tstates_per_scanline = _timing_tstates_per_scanline__default * _timing_tstate_factor;
	_timing_tstates_per_video_frame = _timing_tstates_per_video_frame__default * _timing_tstate_factor;
}

Uint64 timing_get_total_real_time_spent_while_faster() {
	return _fasterTotalRunTimeMs;
}

void timing_apply_tstate_factor(Uint8 factor) {
	if (_timing_is_ready_to_destroy) {
		return;
	}

	Uint8 previousFactor = _timing_tstate_factor;
	_timing_apply_tstate_factor__worker(factor);

	if (factor == 1 && previousFactor > 1) {
		// CPU is resumed to normal speed mode, so add the elapsed time to total time
		struct timeb now;
		ftime(&now);
		Uint64 elapsedFasterTimeMs = (Uint64)(1000.0 * (now.time - _fasterTimingStart.time)
			+ (now.millitm - _fasterTimingStart.millitm));

		// accumulate
		_fasterTotalRunTimeMs += elapsedFasterTimeMs;
	}
	else if (factor > 1 && previousFactor == 1) {
		// CPU has just become faster, so mark the start time
		ftime(&_fasterTimingStart);
	}
}

Uint8 timing_is_faster() {
	return _timing_tstate_factor > 1;
}

void timing_prepare_destroy() {
	_timing_is_ready_to_destroy = 1;
	_timing_apply_tstate_factor__worker(1);
}

double timing_get_actual_milliseconds_per_video_frame() {
	return _timing_actualmilliseconds_per_video_frame;
}

void timing_set_actual_milliseconds_per_video_frame(double actualMsPerVideoFrame) {
	_timing_actualmilliseconds_per_video_frame = actualMsPerVideoFrame;
}

void timing_record_idle_time(Uint64 durationMs) {
	if (timing_is_faster()) {
		_idleTimeWhileCPUFaster += durationMs;
	}
	else {
		_idleTimeAll += durationMs;
	}
}

Uint64 timing_get_total_real_idle_time() {
	return _idleTimeAll;
}

Uint64 timing_get_total_real_idle_time_spent_while_faster() {
	return _idleTimeWhileCPUFaster;
}

void _timing_set_windows_timer_resolution() {
	TIMECAPS tc;
	if (timeGetDevCaps(&tc, sizeof(TIMECAPS)) != TIMERR_NOERROR)
	{
		log_write("Warning: unable to query Windows timer device capabilities");
		return;
	}

	sprintf_s(_debugBuffer, _DEBUG_BUFFER_SIZE - 10, "Windows timer device capabilities: %ums min, %ums max",
		tc.wPeriodMin,
		tc.wPeriodMax);
	log_write(_debugBuffer);

	_windows_timer_min_resolution = min(max(tc.wPeriodMin, TARGET_RESOLUTION_MS), tc.wPeriodMax);
	if (TIMERR_NOERROR != timeBeginPeriod(_windows_timer_min_resolution)) {
		log_write("Warning: unable to begin Windows timer period");
		return;
	}
	
	sprintf_s(_debugBuffer, _DEBUG_BUFFER_SIZE - 10, "Windows timer device resolution set to %ums",
		_windows_timer_min_resolution);
	log_write(_debugBuffer);

	_timer_resolution_set = 1;
}

void timing_start(Uint16 millisecondsPerVideoFrame) {
	// 24 tstates for left border
	// 128 tstates for paper
	// 24 tstates for right border
	// 48 tstates for horizontal retrace
	_timing_tstates_per_scanline__default = 224;

	_timing_reset_overflow();

	_timing_milliseconds_per_video_frame = millisecondsPerVideoFrame;
	timing_set_actual_milliseconds_per_video_frame((double)millisecondsPerVideoFrame);

	_timing_tstates_per_video_frame__default = (Uint64)(SPECTRUM_SCREEN_HEIGHT) * (_timing_tstates_per_scanline__default);

	timing_apply_tstate_factor(1);
	_timing_set_windows_timer_resolution();
}

void timing_destroy() {
	if (_timer_resolution_set) {
		if (TIMERR_NOERROR != timeEndPeriod(_windows_timer_min_resolution)) {
			log_write("Warning: unable to end Windows timer period");
		}
	}
}
