#include "Constants.h"
#include "Cpu.h"
#include "Memory.h"
#include "Io.h"
#include "CpuUtilities.h"
#include "CpuDDCBAndFDCBPrefix.h"


Uint16 cpu_get_absolute_address_indexreg16_plus_disp8(struct instruction* instruction) {
	Uint16* indexReg = cpu_get_indexreg16_ptr(instruction);
	Uint16 address = cpu_compute_absolute_address(*indexReg, instruction->displacementValue);
	return address;
}

void cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(
	struct instruction* instruction, const char* sourceNonNegative, const char* sourceNegative, const char* indexRegName) {

	if ((Sint8)instruction->displacementValue < 0) {
		cpu_set_instruction_disassembled_name_formatted_str(instruction, sourceNegative, cpu_neg8(instruction->displacementValue), UNUSED, indexRegName, UNUSEDSTR);
	}
	else {
		cpu_set_instruction_disassembled_name_formatted_str(instruction, sourceNonNegative, instruction->displacementValue, UNUSED, indexRegName, UNUSEDSTR);
	}
}

// writes to memory and then returns final value
Uint8 cpu_instruction__rlc_indexreg16Ptr_reg8(struct instruction* instruction, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 7);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	// correct result must be present in target reg8 even when ptr is to ROM
	Uint8 existingValue = memory_read8(absoluteAddress);
	Uint16 val16 = (Uint16)(existingValue);
	Uint8 bit7 = val16 >> 7;
	if (bit7) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;

	val16 = val16 << 1;		// bit 0 := 0
	val16 |= bit7;			// bit 0 := bit 7

	Uint8 finalValue = (Uint8)val16;
	memory_write8(absoluteAddress, finalValue);

	if ((finalValue) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(finalValue)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(finalValue)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
	return finalValue;
}

// writes to memory and then returns final value
Uint8 cpu_instruction__rrc_indexreg16Ptr_reg8(struct instruction* instruction, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 7);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	// correct result must be present in target reg8 even when ptr is to ROM
	Uint8 existingValue = memory_read8(absoluteAddress);
	
	Uint8 bit0 = existingValue & 0x01;
	if (bit0) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;

	Uint8 finalValue = existingValue >> 1;
	if (bit0) (finalValue) |= 0x80; else (finalValue) &= 0x7F;
	memory_write8(absoluteAddress, finalValue);

	if ((finalValue) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(finalValue)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(finalValue)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
	return finalValue;
}

// writes to memory and then returns final value
Uint8 cpu_instruction__rl_indexreg16Ptr_reg8(struct instruction* instruction, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 7);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	// correct result must be present in target reg8 even when ptr is to ROM
	Uint8 existingValue = memory_read8(absoluteAddress);

	Uint8 oldCarry = 0;
	if (cpu_regs()->F & CPU_FLAG_C_CARRY) {
		oldCarry = 1;
	}

	Uint16 temp16 = (Uint16)(existingValue);
	Uint8 bit7 = temp16 >> 7;
	if (bit7) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;

	temp16 = temp16 << 1;		// bit 0 := 0
	temp16 |= oldCarry;
	Uint8 finalValue = (Uint8)temp16;
	memory_write8(absoluteAddress, finalValue);

	if ((finalValue) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(finalValue)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(finalValue)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);
	return finalValue;
}

// writes to memory and then returns final value
Uint8 cpu_instruction__rr_indexreg16Ptr_reg8(struct instruction* instruction, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 7);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	// correct result must be present in target reg8 even when ptr is to ROM
	Uint8 existingValue = memory_read8(absoluteAddress);

	Uint8 oldCarry = 0;
	if (cpu_regs()->F & CPU_FLAG_C_CARRY) {
		oldCarry = 1;
	}

	Uint16 temp16 = (Uint16)(existingValue);
	if (temp16 & 1) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;

	temp16 = temp16 >> 1;
	if (oldCarry) temp16 |= 0x80; else; temp16 &= 0x7FF;
	Uint8 finalValue = (Uint8)temp16;
	memory_write8(absoluteAddress, finalValue);

	if ((finalValue) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(finalValue)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(finalValue)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);

	return finalValue;
}

// writes to memory and then returns final value
Uint8 cpu_instruction__sla_indexreg16Ptr_reg8(struct instruction* instruction, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 7);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	// correct result must be present in target reg8 even when ptr is to ROM
	Uint8 existingValue = memory_read8(absoluteAddress);

	if ((existingValue) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;
	Uint8 finalValue = (existingValue) << 1;
	memory_write8(absoluteAddress, finalValue);

	if ((finalValue) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(finalValue)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(finalValue)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);

	return finalValue;
}

// writes to memory and then returns final value
Uint8 cpu_instruction__sra_indexreg16Ptr_reg8(struct instruction* instruction, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 7);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	// correct result must be present in target reg8 even when ptr is to ROM
	Uint8 existingValue = memory_read8(absoluteAddress);

	if ((existingValue) & 1) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;
	Uint8 bit7 = (existingValue) & (1 << 7);
	Uint8 finalValue = (existingValue) >> 1;
	finalValue |= bit7;
	memory_write8(absoluteAddress, finalValue);

	if ((finalValue) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(finalValue)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(finalValue)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);

	return finalValue;
}

// writes to memory and then returns final value
Uint8 cpu_instruction__sll_indexreg16Ptr_reg8(struct instruction* instruction, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 7);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	// correct result must be present in target reg8 even when ptr is to ROM
	Uint8 existingValue = memory_read8(absoluteAddress);

	if ((existingValue) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;
	Uint8 finalValue = (existingValue) << 1;
	(finalValue) |= 1;	// put a 1 in bit0
	memory_write8(absoluteAddress, finalValue);

	if ((finalValue) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(finalValue)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(finalValue)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);

	return finalValue;
}

// writes to memory and then returns final value
Uint8 cpu_instruction__srl_indexreg16Ptr_reg8(struct instruction* instruction, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 7);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	// correct result must be present in target reg8 even when ptr is to ROM
	Uint8 existingValue = memory_read8(absoluteAddress);

	if ((existingValue) & 1) cpu_regs()->F |= CPU_FLAG_C_CARRY; else cpu_regs()->F &= ~CPU_FLAG_C_CARRY;
	Uint8 finalValue = (existingValue) >> 1;
	memory_write8(absoluteAddress, finalValue);

	if ((finalValue) & (1 << 7)) cpu_regs()->F |= CPU_FLAG_S_SIGN; else cpu_regs()->F &= ~CPU_FLAG_S_SIGN;
	if (!(finalValue)) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;
	if (cpu_parity(finalValue)) cpu_regs()->F |= CPU_FLAG_PV_PARITY_OVERFLOW; else cpu_regs()->F &= ~CPU_FLAG_PV_PARITY_OVERFLOW;
	cpu_regs()->F &= ~CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);

	return finalValue;
}

// returns existing value
Uint8 cpu_instruction__bit_indexreg16Ptr(struct instruction* instruction, Uint8 bit, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 4);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	Uint8 existingValue = memory_read8(absoluteAddress);

	if ((existingValue & (1 << bit)) == 0) cpu_regs()->F |= CPU_FLAG_Z_ZERO; else cpu_regs()->F &= ~CPU_FLAG_Z_ZERO;

	cpu_regs()->F |= CPU_FLAG_H_HALF_CARRY;
	cpu_regs()->F &= ~CPU_FLAG_N_SUBTRACT;

	cpu_mark_instruction_executed(instruction);

	return existingValue;
}

// writes to memory and then returns final value
Uint8 cpu_instruction__res_indexreg16Ptr_reg8(struct instruction* instruction, Uint8 bit, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 7);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	// correct result must be present in target reg8 even when ptr is to ROM
	Uint8 existingValue = memory_read8(absoluteAddress);

	Uint8 shiftedBit = 1 << bit;
	Uint8 finalValue = existingValue & ~shiftedBit;
	memory_write8(absoluteAddress, finalValue);

	cpu_mark_instruction_executed(instruction);

	return finalValue;
}

// writes to memory and then returns final value
Uint8 cpu_instruction__set_indexreg16Ptr_reg8(struct instruction* instruction, Uint8 bit, const char* nonNegativeSource, const char* negativeSource) {
	Uint16 absoluteAddress = cpu_get_absolute_address_indexreg16_plus_disp8(instruction);
	const char* indexRegName = cpu_get_indexreg16_name(instruction);

	cpu_finalize_instruction_decode(instruction, 7);
	cpu_set_instruction_disassembled_name_formatted_indexreg16_plus_disp8(instruction, nonNegativeSource, negativeSource, indexRegName);

	if (!cpu_must_execute()) return 0xF3;
	//execution

	// correct result must be present in target reg8 even when ptr is to ROM
	Uint8 existingValue = memory_read8(absoluteAddress);

	Uint8 shiftedBit = 1 << bit;
	Uint8 finalValue = existingValue | shiftedBit;
	memory_write8(absoluteAddress, finalValue);

	cpu_mark_instruction_executed(instruction);

	return finalValue;
}




void cpu_run_DDCB_and_FDCB_prefix(struct instruction* instruction) {
	Uint8 temp8;

	switch (instruction->opcodeValue) {
	case 0x00:	// rlc (indexreg16+disp8), b
		temp8 = cpu_instruction__rlc_indexreg16Ptr_reg8(instruction, "rlc (/s0+/n0), B", "rlc (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x01:	// rlc (indexreg16+disp8), c
		temp8 = cpu_instruction__rlc_indexreg16Ptr_reg8(instruction, "rlc (/s0+/n0), C", "rlc (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x02:	// rlc (indexreg16+disp8), d
		temp8 = cpu_instruction__rlc_indexreg16Ptr_reg8(instruction, "rlc (/s0+/n0), D", "rlc (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x03:	// rlc (indexreg16+disp8), e
		temp8 = cpu_instruction__rlc_indexreg16Ptr_reg8(instruction, "rlc (/s0+/n0), E", "rlc (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x04:	// rlc (indexreg16+disp8), h
		temp8 = cpu_instruction__rlc_indexreg16Ptr_reg8(instruction, "rlc (/s0+/n0), H", "rlc (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x05:	// rlc (indexreg16+disp8), l
		temp8 = cpu_instruction__rlc_indexreg16Ptr_reg8(instruction, "rlc (/s0+/n0), L", "rlc (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x06:	// rlc (indexreg16+disp8)
		cpu_instruction__rlc_indexreg16Ptr_reg8(instruction, "rlc (/s0+/n0)", "rlc (/s0-/n0)");
		break;
	case 0x07:	// rlc (indexreg16+disp8), a
		temp8 = cpu_instruction__rlc_indexreg16Ptr_reg8(instruction, "rlc (/s0+/n0), A", "rlc (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x08:	// rrc (indexreg16+disp8), b
		temp8 = cpu_instruction__rrc_indexreg16Ptr_reg8(instruction, "rrc (/s0+/n0), B", "rrc (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x09:	// rrc (indexreg16+disp8), c
		temp8 = cpu_instruction__rrc_indexreg16Ptr_reg8(instruction, "rrc (/s0+/n0), C", "rrc (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x0A:	// rrc (indexreg16+disp8), d
		temp8 = cpu_instruction__rrc_indexreg16Ptr_reg8(instruction, "rrc (/s0+/n0), D", "rrc (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x0B:	// rrc (indexreg16+disp8), e
		temp8 = cpu_instruction__rrc_indexreg16Ptr_reg8(instruction, "rrc (/s0+/n0), E", "rrc (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x0C:	// rrc (indexreg16+disp8), h
		temp8 = cpu_instruction__rrc_indexreg16Ptr_reg8(instruction, "rrc (/s0+/n0), H", "rrc (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x0D:	// rrc (indexreg16+disp8), l
		temp8 = cpu_instruction__rrc_indexreg16Ptr_reg8(instruction, "rrc (/s0+/n0), L", "rrc (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x0E:	// rrc (indexreg16+disp8)
		cpu_instruction__rrc_indexreg16Ptr_reg8(instruction, "rrc (/s0+/n0)", "rrc (/s0-/n0)");
		break;
	case 0x0F:	// rrc (indexreg16+disp8), a
		temp8 = cpu_instruction__rrc_indexreg16Ptr_reg8(instruction, "rrc (/s0+/n0), A", "rrc (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x10:	// rl (indexreg16+disp8), b
		temp8 = cpu_instruction__rl_indexreg16Ptr_reg8(instruction, "rl (/s0+/n0), B", "rl (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x11:	// rl (indexreg16+disp8), c
		temp8 = cpu_instruction__rl_indexreg16Ptr_reg8(instruction, "rl (/s0+/n0), C", "rl (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x12:	// rl (indexreg16+disp8), d
		temp8 = cpu_instruction__rl_indexreg16Ptr_reg8(instruction, "rl (/s0+/n0), D", "rl (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x13:	// rl (indexreg16+disp8), e
		temp8 = cpu_instruction__rl_indexreg16Ptr_reg8(instruction, "rl (/s0+/n0), E", "rl (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x14:	// rl (indexreg16+disp8), h
		temp8 = cpu_instruction__rl_indexreg16Ptr_reg8(instruction, "rl (/s0+/n0), H", "rl (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x15:	// rl (indexreg16+disp8), l
		temp8 = cpu_instruction__rl_indexreg16Ptr_reg8(instruction, "rl (/s0+/n0), L", "rl (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x16:	// rl (indexreg16+disp8)
		cpu_instruction__rl_indexreg16Ptr_reg8(instruction, "rl (/s0+/n0)", "rl (/s0-/n0)");
		break;
	case 0x17:	// rl (indexreg16+disp8), a
		temp8 = cpu_instruction__rl_indexreg16Ptr_reg8(instruction, "rl (/s0+/n0), A", "rl (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x18:	// rr (indexreg16+disp8), b
		temp8 = cpu_instruction__rr_indexreg16Ptr_reg8(instruction, "rr (/s0+/n0), B", "rr (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x19:	// rr (indexreg16+disp8), c
		temp8 = cpu_instruction__rr_indexreg16Ptr_reg8(instruction, "rr (/s0+/n0), C", "rr (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x1A:	// rr (indexreg16+disp8), d
		temp8 = cpu_instruction__rr_indexreg16Ptr_reg8(instruction, "rr (/s0+/n0), D", "rr (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x1B:	// rr (indexreg16+disp8), e
		temp8 = cpu_instruction__rr_indexreg16Ptr_reg8(instruction, "rr (/s0+/n0), E", "rr (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x1C:	// rr (indexreg16+disp8), h
		temp8 = cpu_instruction__rr_indexreg16Ptr_reg8(instruction, "rr (/s0+/n0), H", "rr (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x1D:	// rr (indexreg16+disp8), l
		temp8 = cpu_instruction__rr_indexreg16Ptr_reg8(instruction, "rr (/s0+/n0), L", "rr (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x1E:	// rr (indexreg16+disp8)
		temp8 = cpu_instruction__rr_indexreg16Ptr_reg8(instruction, "rr (/s0+/n0)", "rr (/s0-/n0)");
		break;
	case 0x1F:	// rr (indexreg16+disp8), a
		temp8 = cpu_instruction__rr_indexreg16Ptr_reg8(instruction, "rr (/s0+/n0), A", "rr (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x20:	// sla (indexreg16+disp8), b
		temp8 = cpu_instruction__sla_indexreg16Ptr_reg8(instruction, "sla (/s0+/n0), B", "sla (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x21:	// sla (indexreg16+disp8), c
		temp8 = cpu_instruction__sla_indexreg16Ptr_reg8(instruction, "sla (/s0+/n0), C", "sla (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x22:	// sla (indexreg16+disp8), d
		temp8 = cpu_instruction__sla_indexreg16Ptr_reg8(instruction, "sla (/s0+/n0), D", "sla (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x23:	// sla (indexreg16+disp8), e
		temp8 = cpu_instruction__sla_indexreg16Ptr_reg8(instruction, "sla (/s0+/n0), E", "sla (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x24:	// sla (indexreg16+disp8), h
		temp8 = cpu_instruction__sla_indexreg16Ptr_reg8(instruction, "sla (/s0+/n0), H", "sla (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x25:	// sla (indexreg16+disp8), l
		temp8 = cpu_instruction__sla_indexreg16Ptr_reg8(instruction, "sla (/s0+/n0), L", "sla (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x26:	// sla (indexreg16+disp8)
		temp8 = cpu_instruction__sla_indexreg16Ptr_reg8(instruction, "sla (/s0+/n0)", "sla (/s0-/n0)");
		break;
	case 0x27:	// sla (indexreg16+disp8), a
		temp8 = cpu_instruction__sla_indexreg16Ptr_reg8(instruction, "sla (/s0+/n0), A", "sla (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x28:	// sra (indexreg16+disp8), b
		temp8 = cpu_instruction__sra_indexreg16Ptr_reg8(instruction, "sra (/s0+/n0), B", "sra (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x29:	// sra (indexreg16+disp8), c
		temp8 = cpu_instruction__sra_indexreg16Ptr_reg8(instruction, "sra (/s0+/n0), C", "sra (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x2A:	// sra (indexreg16+disp8), d
		temp8 = cpu_instruction__sra_indexreg16Ptr_reg8(instruction, "sra (/s0+/n0), D", "sra (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x2B:	// sra (indexreg16+disp8), e
		temp8 = cpu_instruction__sra_indexreg16Ptr_reg8(instruction, "sra (/s0+/n0), E", "sra (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x2C:	// sra (indexreg16+disp8), h
		temp8 = cpu_instruction__sra_indexreg16Ptr_reg8(instruction, "sra (/s0+/n0), H", "sra (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x2D:	// sra (indexreg16+disp8), l
		temp8 = cpu_instruction__sra_indexreg16Ptr_reg8(instruction, "sra (/s0+/n0), L", "sra (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x2E:	// sra (indexreg16+disp8)
		temp8 = cpu_instruction__sra_indexreg16Ptr_reg8(instruction, "sra (/s0+/n0)", "sra (/s0-/n0)");
		break;
	case 0x2F:	// sra (indexreg16+disp8), a
		temp8 = cpu_instruction__sra_indexreg16Ptr_reg8(instruction, "sra (/s0+/n0), A", "sra (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x30:	// sll (indexreg16+disp8), b
		temp8 = cpu_instruction__sll_indexreg16Ptr_reg8(instruction, "sll (/s0+/n0), B", "sll (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x31:	// sll (indexreg16+disp8), c
		temp8 = cpu_instruction__sll_indexreg16Ptr_reg8(instruction, "sll (/s0+/n0), C", "sll (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x32:	// sll (indexreg16+disp8), d
		temp8 = cpu_instruction__sll_indexreg16Ptr_reg8(instruction, "sll (/s0+/n0), D", "sll (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x33:	// sll (indexreg16+disp8), e
		temp8 = cpu_instruction__sll_indexreg16Ptr_reg8(instruction, "sll (/s0+/n0), E", "sll (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x34:	// sll (indexreg16+disp8), h
		temp8 = cpu_instruction__sll_indexreg16Ptr_reg8(instruction, "sll (/s0+/n0), H", "sll (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x35:	// sll (indexreg16+disp8), l
		temp8 = cpu_instruction__sll_indexreg16Ptr_reg8(instruction, "sll (/s0+/n0), L", "sll (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x36:	// sll (indexreg16+disp8)
		temp8 = cpu_instruction__sll_indexreg16Ptr_reg8(instruction, "sll (/s0+/n0)", "sll (/s0-/n0)");
		break;
	case 0x37:	// sll (indexreg16+disp8), a
		temp8 = cpu_instruction__sll_indexreg16Ptr_reg8(instruction, "sll (/s0+/n0), A", "sll (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x38:	// srl (indexreg16+disp8), b
		temp8 = cpu_instruction__srl_indexreg16Ptr_reg8(instruction, "srl (/s0+/n0), B", "srl (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x39:	// srl (indexreg16+disp8), c
		temp8 = cpu_instruction__srl_indexreg16Ptr_reg8(instruction, "srl (/s0+/n0), C", "srl (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x3A:	// srl (indexreg16+disp8), d
		temp8 = cpu_instruction__srl_indexreg16Ptr_reg8(instruction, "srl (/s0+/n0), D", "srl (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x3B:	// srl (indexreg16+disp8), e
		temp8 = cpu_instruction__srl_indexreg16Ptr_reg8(instruction, "srl (/s0+/n0), E", "srl (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x3C:	// srl (indexreg16+disp8), h
		temp8 = cpu_instruction__srl_indexreg16Ptr_reg8(instruction, "srl (/s0+/n0), H", "srl (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x3D:	// srl (indexreg16+disp8), l
		temp8 = cpu_instruction__srl_indexreg16Ptr_reg8(instruction, "srl (/s0+/n0), L", "srl (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x3E:	// srl (indexreg16+disp8)
		temp8 = cpu_instruction__srl_indexreg16Ptr_reg8(instruction, "srl (/s0+/n0)", "srl (/s0-/n0)");
		break;
	case 0x3F:	// srl (indexreg16+disp8), a
		temp8 = cpu_instruction__srl_indexreg16Ptr_reg8(instruction, "srl (/s0+/n0), A", "srl (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x40:	// bit 0, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 0, "bit 0, (/s0+/n0)", "bit 0, (/s0-/n0)");
		break;
	case 0x41:	// bit 0, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 0, "bit 0, (/s0+/n0)", "bit 0, (/s0-/n0)");
		break;
	case 0x42:	// bit 0, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 0, "bit 0, (/s0+/n0)", "bit 0, (/s0-/n0)");
		break;
	case 0x43:	// bit 0, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 0, "bit 0, (/s0+/n0)", "bit 0, (/s0-/n0)");
		break;
	case 0x44:	// bit 0, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 0, "bit 0, (/s0+/n0)", "bit 0, (/s0-/n0)");
		break;
	case 0x45:	// bit 0, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 0, "bit 0, (/s0+/n0)", "bit 0, (/s0-/n0)");
		break;
	case 0x46:	// bit 0, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 0, "bit 0, (/s0+/n0)", "bit 0, (/s0-/n0)");
		break;
	case 0x47:	// bit 0, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 0, "bit 0, (/s0+/n0)", "bit 0, (/s0-/n0)");
		break;
	case 0x48:	// bit 1, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 1, "bit 1, (/s0+/n0)", "bit 1, (/s0-/n0)");
		break;
	case 0x49:	// bit 1, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 1, "bit 1, (/s0+/n0)", "bit 1, (/s0-/n0)");
		break;
	case 0x4A:	// bit 1, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 1, "bit 1, (/s0+/n0)", "bit 1, (/s0-/n0)");
		break;
	case 0x4B:	// bit 1, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 1, "bit 1, (/s0+/n0)", "bit 1, (/s0-/n0)");
		break;
	case 0x4C:	// bit 1, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 1, "bit 1, (/s0+/n0)", "bit 1, (/s0-/n0)");
		break;
	case 0x4D:	// bit 1, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 1, "bit 1, (/s0+/n0)", "bit 1, (/s0-/n0)");
		break;
	case 0x4E:	// bit 1, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 1, "bit 1, (/s0+/n0)", "bit 1, (/s0-/n0)");
		break;
	case 0x4F:	// bit 1, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 1, "bit 1, (/s0+/n0)", "bit 1, (/s0-/n0)");
		break;
	case 0x50:	// bit 2, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 2, "bit 2, (/s0+/n0)", "bit 2, (/s0-/n0)");
		break;
	case 0x51:	// bit 2, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 2, "bit 2, (/s0+/n0)", "bit 2, (/s0-/n0)");
		break;
	case 0x52:	// bit 2, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 2, "bit 2, (/s0+/n0)", "bit 2, (/s0-/n0)");
		break;
	case 0x53:	// bit 2, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 2, "bit 2, (/s0+/n0)", "bit 2, (/s0-/n0)");
		break;
	case 0x54:	// bit 2, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 2, "bit 2, (/s0+/n0)", "bit 2, (/s0-/n0)");
		break;
	case 0x55:	// bit 2, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 2, "bit 2, (/s0+/n0)", "bit 2, (/s0-/n0)");
		break;
	case 0x56:	// bit 2, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 2, "bit 2, (/s0+/n0)", "bit 2, (/s0-/n0)");
		break;
	case 0x57:	// bit 2, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 2, "bit 2, (/s0+/n0)", "bit 2, (/s0-/n0)");
		break;
	case 0x58:	// bit 3, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 3, "bit 3, (/s0+/n0)", "bit 3, (/s0-/n0)");
		break;
	case 0x59:	// bit 3, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 3, "bit 3, (/s0+/n0)", "bit 3, (/s0-/n0)");
		break;
	case 0x5A:	// bit 3, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 3, "bit 3, (/s0+/n0)", "bit 3, (/s0-/n0)");
		break;
	case 0x5B:	// bit 3, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 3, "bit 3, (/s0+/n0)", "bit 3, (/s0-/n0)");
		break;
	case 0x5C:	// bit 3, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 3, "bit 3, (/s0+/n0)", "bit 3, (/s0-/n0)");
		break;
	case 0x5D:	// bit 3, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 3, "bit 3, (/s0+/n0)", "bit 3, (/s0-/n0)");
		break;
	case 0x5E:	// bit 3, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 3, "bit 3, (/s0+/n0)", "bit 3, (/s0-/n0)");
		break;
	case 0x5F:	// bit 3, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 3, "bit 3, (/s0+/n0)", "bit 3, (/s0-/n0)");
		break;
	case 0x60:	// bit 4, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 4, "bit 4, (/s0+/n0)", "bit 4, (/s0-/n0)");
		break;
	case 0x61:	// bit 4, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 4, "bit 4, (/s0+/n0)", "bit 4, (/s0-/n0)");
		break;
	case 0x62:	// bit 4, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 4, "bit 4, (/s0+/n0)", "bit 4, (/s0-/n0)");
		break;
	case 0x63:	// bit 4, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 4, "bit 4, (/s0+/n0)", "bit 4, (/s0-/n0)");
		break;
	case 0x64:	// bit 4, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 4, "bit 4, (/s0+/n0)", "bit 4, (/s0-/n0)");
		break;
	case 0x65:	// bit 4, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 4, "bit 4, (/s0+/n0)", "bit 4, (/s0-/n0)");
		break;
	case 0x66:	// bit 4, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 4, "bit 4, (/s0+/n0)", "bit 4, (/s0-/n0)");
		break;
	case 0x67:	// bit 4, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 4, "bit 4, (/s0+/n0)", "bit 4, (/s0-/n0)");
		break;
	case 0x68:	// bit 5, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 5, "bit 5, (/s0+/n0)", "bit 5, (/s0-/n0)");
		break;
	case 0x69:	// bit 5, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 5, "bit 5, (/s0+/n0)", "bit 5, (/s0-/n0)");
		break;
	case 0x6A:	// bit 5, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 5, "bit 5, (/s0+/n0)", "bit 5, (/s0-/n0)");
		break;
	case 0x6B:	// bit 5, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 5, "bit 5, (/s0+/n0)", "bit 5, (/s0-/n0)");
		break;
	case 0x6C:	// bit 5, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 5, "bit 5, (/s0+/n0)", "bit 5, (/s0-/n0)");
		break;
	case 0x6D:	// bit 5, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 5, "bit 5, (/s0+/n0)", "bit 5, (/s0-/n0)");
		break;
	case 0x6E:	// bit 5, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 5, "bit 5, (/s0+/n0)", "bit 5, (/s0-/n0)");
		break;
	case 0x6F:	// bit 5, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 5, "bit 5, (/s0+/n0)", "bit 5, (/s0-/n0)");
		break;
	case 0x70:	// bit 6, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 6, "bit 6, (/s0+/n0)", "bit 6, (/s0-/n0)");
		break;
	case 0x71:	// bit 6, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 6, "bit 6, (/s0+/n0)", "bit 6, (/s0-/n0)");
		break;
	case 0x72:	// bit 6, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 6, "bit 6, (/s0+/n0)", "bit 6, (/s0-/n0)");
		break;
	case 0x73:	// bit 6, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 6, "bit 6, (/s0+/n0)", "bit 6, (/s0-/n0)");
		break;
	case 0x74:	// bit 6, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 6, "bit 6, (/s0+/n0)", "bit 6, (/s0-/n0)");
		break;
	case 0x75:	// bit 6, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 6, "bit 6, (/s0+/n0)", "bit 6, (/s0-/n0)");
		break;
	case 0x76:	// bit 6, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 6, "bit 6, (/s0+/n0)", "bit 6, (/s0-/n0)");
		break;
	case 0x77:	// bit 6, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 6, "bit 6, (/s0+/n0)", "bit 6, (/s0-/n0)");
		break;
	case 0x78:	// bit 7, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 7, "bit 7, (/s0+/n0)", "bit 7, (/s0-/n0)");
		break;
	case 0x79:	// bit 7, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 7, "bit 7, (/s0+/n0)", "bit 7, (/s0-/n0)");
		break;
	case 0x7A:	// bit 7, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 7, "bit 7, (/s0+/n0)", "bit 7, (/s0-/n0)");
		break;
	case 0x7B:	// bit 7, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 7, "bit 7, (/s0+/n0)", "bit 7, (/s0-/n0)");
		break;
	case 0x7C:	// bit 7, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 7, "bit 7, (/s0+/n0)", "bit 7, (/s0-/n0)");
		break;
	case 0x7D:	// bit 7, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 7, "bit 7, (/s0+/n0)", "bit 7, (/s0-/n0)");
		break;
	case 0x7E:	// bit 7, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 7, "bit 7, (/s0+/n0)", "bit 7, (/s0-/n0)");
		break;
	case 0x7F:	// bit 7, (indexreg16+disp8)
		cpu_instruction__bit_indexreg16Ptr(instruction, 7, "bit 7, (/s0+/n0)", "bit 7, (/s0-/n0)");
		break;
	case 0x80:	// res 0, (indexreg16+disp8), b
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 0, "res 0, (/s0+/n0), B", "res 0, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x81:	// res 0, (indexreg16+disp8), c
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 0, "res 0, (/s0+/n0), C", "res 0, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x82:	// res 0, (indexreg16+disp8), d
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 0, "res 0, (/s0+/n0), D", "res 0, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x83:	// res 0, (indexreg16+disp8), e
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 0, "res 0, (/s0+/n0), E", "res 0, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x84:	// res 0, (indexreg16+disp8), h
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 0, "res 0, (/s0+/n0), H", "res 0, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x85:	// res 0, (indexreg16+disp8), l
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 0, "res 0, (/s0+/n0), L", "res 0, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x86:	// res 0, (indexreg16+disp8)
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 0, "res 0, (/s0+/n0)", "res 0, (/s0-/n0)");
		break;
	case 0x87:	// res 0, (indexreg16+disp8), a
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 0, "res 0, (/s0+/n0), A", "res 0, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x88:	// res 1, (indexreg16+disp8), b
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 1, "res 1, (/s0+/n0), B", "res 1, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x89:	// res 1, (indexreg16+disp8), c
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 1, "res 1, (/s0+/n0), C", "res 1, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x8A:	// res 1, (indexreg16+disp8), d
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 1, "res 1, (/s0+/n0), D", "res 1, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x8B:	// res 1, (indexreg16+disp8), e
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 1, "res 1, (/s0+/n0), E", "res 1, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x8C:	// res 1, (indexreg16+disp8), h
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 1, "res 1, (/s0+/n0), H", "res 1, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x8D:	// res 1, (indexreg16+disp8), l
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 1, "res 1, (/s0+/n0), L", "res 1, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x8E:	// res 1, (indexreg16+disp8)
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 1, "res 1, (/s0+/n0)", "res 1, (/s0-/n0)");
		break;
	case 0x8F:	// res 1, (indexreg16+disp8), a
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 1, "res 1, (/s0+/n0), A", "res 1, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x90:	// res 2, (indexreg16+disp8), b
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 2, "res 2, (/s0+/n0), B", "res 2, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x91:	// res 2, (indexreg16+disp8), c
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 2, "res 2, (/s0+/n0), C", "res 2, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x92:	// res 2, (indexreg16+disp8), d
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 2, "res 2, (/s0+/n0), D", "res 2, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x93:	// res 2, (indexreg16+disp8), e
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 2, "res 2, (/s0+/n0), E", "res 2, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x94:	// res 2, (indexreg16+disp8), h
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 2, "res 2, (/s0+/n0), H", "res 2, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x95:	// res 2, (indexreg16+disp8), l
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 2, "res 2, (/s0+/n0), L", "res 2, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x96:	// res 2, (indexreg16+disp8)
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 2, "res 2, (/s0+/n0)", "res 2, (/s0-/n0)");
		break;
	case 0x97:	// res 2, (indexreg16+disp8), a
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 2, "res 2, (/s0+/n0), A", "res 2, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0x98:	// res 3, (indexreg16+disp8), b
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 3, "res 3, (/s0+/n0), B", "res 3, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0x99:	// res 3, (indexreg16+disp8), c
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 3, "res 3, (/s0+/n0), C", "res 3, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0x9A:	// res 3, (indexreg16+disp8), d
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 3, "res 3, (/s0+/n0), D", "res 3, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0x9B:	// res 3, (indexreg16+disp8), e
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 3, "res 3, (/s0+/n0), E", "res 3, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0x9C:	// res 3, (indexreg16+disp8), h
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 3, "res 3, (/s0+/n0), H", "res 3, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0x9D:	// res 3, (indexreg16+disp8), l
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 3, "res 3, (/s0+/n0), L", "res 3, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0x9E:	// res 3, (indexreg16+disp8)
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 3, "res 3, (/s0+/n0)", "res 3, (/s0-/n0)");
		break;
	case 0x9F:	// res 3, (indexreg16+disp8), a
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 3, "res 3, (/s0+/n0), A", "res 3, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xA0:	// res 4, (indexreg16+disp8), b
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 4, "res 4, (/s0+/n0), B", "res 4, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xA1:	// res 4, (indexreg16+disp8), c
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 4, "res 4, (/s0+/n0), C", "res 4, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xA2:	// res 4, (indexreg16+disp8), d
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 4, "res 4, (/s0+/n0), D", "res 4, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xA3:	// res 4, (indexreg16+disp8), e
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 4, "res 4, (/s0+/n0), E", "res 4, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xA4:	// res 4, (indexreg16+disp8), h
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 4, "res 4, (/s0+/n0), H", "res 4, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xA5:	// res 4, (indexreg16+disp8), l
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 4, "res 4, (/s0+/n0), L", "res 4, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xA6:	// res 4, (indexreg16+disp8)
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 4, "res 4, (/s0+/n0)", "res 4, (/s0-/n0)");
		break;
	case 0xA7:	// res 4, (indexreg16+disp8), a
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 4, "res 4, (/s0+/n0), A", "res 4, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xA8:	// res 5, (indexreg16+disp8), b
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 5, "res 5, (/s0+/n0), B", "res 5, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xA9:	// res 5, (indexreg16+disp8), c
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 5, "res 5, (/s0+/n0), C", "res 5, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xAA:	// res 5, (indexreg16+disp8), d
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 5, "res 5, (/s0+/n0), D", "res 5, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xAB:	// res 5, (indexreg16+disp8), e
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 5, "res 5, (/s0+/n0), E", "res 5, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xAC:	// res 5, (indexreg16+disp8), h
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 5, "res 5, (/s0+/n0), H", "res 5, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xAD:	// res 5, (indexreg16+disp8), l
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 5, "res 5, (/s0+/n0), L", "res 5, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xAE:	// res 5, (indexreg16+disp8)
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 5, "res 5, (/s0+/n0)", "res 5, (/s0-/n0)");
		break;
	case 0xAF:	// res 5, (indexreg16+disp8), a
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 5, "res 5, (/s0+/n0), A", "res 5, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xB0:	// res 6, (indexreg16+disp8), b
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 6, "res 6, (/s0+/n0), B", "res 6, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xB1:	// res 6, (indexreg16+disp8), c
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 6, "res 6, (/s0+/n0), C", "res 6, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xB2:	// res 6, (indexreg16+disp8), d
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 6, "res 6, (/s0+/n0), D", "res 6, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xB3:	// res 6, (indexreg16+disp8), e
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 6, "res 6, (/s0+/n0), E", "res 6, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xB4:	// res 6, (indexreg16+disp8), h
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 6, "res 6, (/s0+/n0), H", "res 6, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xB5:	// res 6, (indexreg16+disp8), l
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 6, "res 6, (/s0+/n0), L", "res 6, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xB6:	// res 6, (indexreg16+disp8)
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 6, "res 6, (/s0+/n0)", "res 6, (/s0-/n0)");
		break;
	case 0xB7:	// res 6, (indexreg16+disp8), a
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 6, "res 6, (/s0+/n0), A", "res 6, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xB8:	// res 7, (indexreg16+disp8), b
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 7, "res 7, (/s0+/n0), B", "res 7, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xB9:	// res 7, (indexreg16+disp8), c
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 7, "res 7, (/s0+/n0), C", "res 7, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xBA:	// res 7, (indexreg16+disp8), d
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 7, "res 7, (/s0+/n0), D", "res 7, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xBB:	// res 7, (indexreg16+disp8), e
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 7, "res 7, (/s0+/n0), E", "res 7, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xBC:	// res 7, (indexreg16+disp8), h
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 7, "res 7, (/s0+/n0), H", "res 7, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xBD:	// res 7, (indexreg16+disp8), l
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 7, "res 7, (/s0+/n0), L", "res 7, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xBE:	// res 7, (indexreg16+disp8)
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 7, "res 7, (/s0+/n0)", "res 7, (/s0-/n0)");
		break;
	case 0xBF:	// res 7, (indexreg16+disp8), a
		temp8 = cpu_instruction__res_indexreg16Ptr_reg8(instruction, 7, "res 7, (/s0+/n0), A", "res 7, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xC0:	// set 0, (indexreg16+disp8), b
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 0, "set 0, (/s0+/n0), B", "set 0, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xC1:	// set 0, (indexreg16+disp8), c
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 0, "set 0, (/s0+/n0), C", "set 0, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xC2:	// set 0, (indexreg16+disp8), d
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 0, "set 0, (/s0+/n0), D", "set 0, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xC3:	// set 0, (indexreg16+disp8), e
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 0, "set 0, (/s0+/n0), E", "set 0, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xC4:	// set 0, (indexreg16+disp8), h
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 0, "set 0, (/s0+/n0), H", "set 0, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xC5:	// set 0, (indexreg16+disp8), l
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 0, "set 0, (/s0+/n0), L", "set 0, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xC6:	// set 0, (indexreg16+disp8)
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 0, "set 0, (/s0+/n0)", "set 0, (/s0-/n0)");
		break;
	case 0xC7:	// set 0, (indexreg16+disp8), a
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 0, "set 0, (/s0+/n0), A", "set 0, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xC8:	// set 1, (indexreg16+disp8), b
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 1, "set 1, (/s0+/n0), B", "set 1, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xC9:	// set 1, (indexreg16+disp8), c
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 1, "set 1, (/s0+/n0), C", "set 1, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xCA:	// set 1, (indexreg16+disp8), d
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 1, "set 1, (/s0+/n0), D", "set 1, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xCB:	// set 1, (indexreg16+disp8), e
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 1, "set 1, (/s0+/n0), E", "set 1, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xCC:	// set 1, (indexreg16+disp8), h
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 1, "set 1, (/s0+/n0), H", "set 1, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xCD:	// set 1, (indexreg16+disp8), l
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 1, "set 1, (/s0+/n0), L", "set 1, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xCE:	// set 1, (indexreg16+disp8)
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 1, "set 1, (/s0+/n0)", "set 1, (/s0-/n0)");
		break;
	case 0xCF:	// set 1, (indexreg16+disp8), a
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 1, "set 1, (/s0+/n0), A", "set 1, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xD0:	// set 2, (indexreg16+disp8), b
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 2, "set 2, (/s0+/n0), B", "set 2, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xD1:	// set 2, (indexreg16+disp8), c
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 2, "set 2, (/s0+/n0), C", "set 2, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xD2:	// set 2, (indexreg16+disp8), d
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 2, "set 2, (/s0+/n0), D", "set 2, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xD3:	// set 2, (indexreg16+disp8), e
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 2, "set 2, (/s0+/n0), E", "set 2, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xD4:	// set 2, (indexreg16+disp8), h
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 2, "set 2, (/s0+/n0), H", "set 2, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xD5:	// set 2, (indexreg16+disp8), l
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 2, "set 2, (/s0+/n0), L", "set 2, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xD6:	// set 2, (indexreg16+disp8)
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 2, "set 2, (/s0+/n0)", "set 2, (/s0-/n0)");
		break;
	case 0xD7:	// set 2, (indexreg16+disp8), a
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 2, "set 2, (/s0+/n0), A", "set 2, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xD8:	// set 3, (indexreg16+disp8), b
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 3, "set 3, (/s0+/n0), B", "set 3, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xD9:	// set 3, (indexreg16+disp8), c
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 3, "set 3, (/s0+/n0), C", "set 3, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xDA:	// set 3, (indexreg16+disp8), d
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 3, "set 3, (/s0+/n0), D", "set 3, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xDB:	// set 3, (indexreg16+disp8), e
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 3, "set 3, (/s0+/n0), E", "set 3, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xDC:	// set 3, (indexreg16+disp8), h
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 3, "set 3, (/s0+/n0), H", "set 3, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xDD:	// set 3, (indexreg16+disp8), l
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 3, "set 3, (/s0+/n0), L", "set 3, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xDE:	// set 3, (indexreg16+disp8)
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 3, "set 3, (/s0+/n0)", "set 3, (/s0-/n0)");
		break;
	case 0xDF:	// set 3, (indexreg16+disp8), a
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 3, "set 3, (/s0+/n0), A", "set 3, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xE0:	// set 4, (indexreg16+disp8), b
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 4, "set 4, (/s0+/n0), B", "set 4, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xE1:	// set 4, (indexreg16+disp8), c
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 4, "set 4, (/s0+/n0), C", "set 4, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xE2:	// set 4, (indexreg16+disp8), d
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 4, "set 4, (/s0+/n0), D", "set 4, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xE3:	// set 4, (indexreg16+disp8), e
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 4, "set 4, (/s0+/n0), E", "set 4, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xE4:	// set 4, (indexreg16+disp8), h
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 4, "set 4, (/s0+/n0), H", "set 4, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xE5:	// set 4, (indexreg16+disp8), l
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 4, "set 4, (/s0+/n0), L", "set 4, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xE6:	// set 4, (indexreg16+disp8)
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 4, "set 4, (/s0+/n0)", "set 4, (/s0-/n0)");
		break;
	case 0xE7:	// set 4, (indexreg16+disp8), a
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 4, "set 4, (/s0+/n0), A", "set 4, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xE8:	// set 5, (indexreg16+disp8), b
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 5, "set 5, (/s0+/n0), B", "set 5, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xE9:	// set 5, (indexreg16+disp8), c
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 5, "set 5, (/s0+/n0), C", "set 5, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xEA:	// set 5, (indexreg16+disp8), d
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 5, "set 5, (/s0+/n0), D", "set 5, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xEB:	// set 5, (indexreg16+disp8), e
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 5, "set 5, (/s0+/n0), E", "set 5, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xEC:	// set 5, (indexreg16+disp8), h
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 5, "set 5, (/s0+/n0), H", "set 5, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xED:	// set 5, (indexreg16+disp8), l
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 5, "set 5, (/s0+/n0), L", "set 5, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xEE:	// set 5, (indexreg16+disp8)
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 5, "set 5, (/s0+/n0)", "set 5, (/s0-/n0)");
		break;
	case 0xEF:	// set 5, (indexreg16+disp8), a
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 5, "set 5, (/s0+/n0), A", "set 5, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xF0:	// set 6, (indexreg16+disp8), b
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 6, "set 6, (/s0+/n0), B", "set 6, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xF1:	// set 6, (indexreg16+disp8), c
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 6, "set 6, (/s0+/n0), C", "set 6, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xF2:	// set 6, (indexreg16+disp8), d
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 6, "set 6, (/s0+/n0), D", "set 6, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xF3:	// set 6, (indexreg16+disp8), e
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 6, "set 6, (/s0+/n0), E", "set 6, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xF4:	// set 6, (indexreg16+disp8), h
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 6, "set 6, (/s0+/n0), H", "set 6, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xF5:	// set 6, (indexreg16+disp8), l
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 6, "set 6, (/s0+/n0), L", "set 6, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xF6:	// set 6, (indexreg16+disp8)
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 6, "set 6, (/s0+/n0)", "set 6, (/s0-/n0)");
		break;
	case 0xF7:	// set 6, (indexreg16+disp8), a
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 6, "set 6, (/s0+/n0), A", "set 6, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;
	case 0xF8:	// set 7, (indexreg16+disp8), b
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 7, "set 7, (/s0+/n0), B", "set 7, (/s0-/n0), B");
		if (!cpu_must_execute()) break;
		cpu_regs()->B = temp8;
		break;
	case 0xF9:	// set 7, (indexreg16+disp8), c
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 7, "set 7, (/s0+/n0), C", "set 7, (/s0-/n0), C");
		if (!cpu_must_execute()) break;
		cpu_regs()->C = temp8;
		break;
	case 0xFA:	// set 7, (indexreg16+disp8), d
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 7, "set 7, (/s0+/n0), D", "set 7, (/s0-/n0), D");
		if (!cpu_must_execute()) break;
		cpu_regs()->D = temp8;
		break;
	case 0xFB:	// set 7, (indexreg16+disp8), e
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 7, "set 7, (/s0+/n0), E", "set 7, (/s0-/n0), E");
		if (!cpu_must_execute()) break;
		cpu_regs()->E = temp8;
		break;
	case 0xFC:	// set 7, (indexreg16+disp8), h
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 7, "set 7, (/s0+/n0), H", "set 7, (/s0-/n0), H");
		if (!cpu_must_execute()) break;
		cpu_regs()->H = temp8;
		break;
	case 0xFD:	// set 7, (indexreg16+disp8), l
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 7, "set 7, (/s0+/n0), L", "set 7, (/s0-/n0), L");
		if (!cpu_must_execute()) break;
		cpu_regs()->L = temp8;
		break;
	case 0xFE:	// set 7, (indexreg16+disp8)
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 7, "set 7, (/s0+/n0)", "set 7, (/s0-/n0)");
		break;
	case 0xFF:	// set 7, (indexreg16+disp8), a
		temp8 = cpu_instruction__set_indexreg16Ptr_reg8(instruction, 7, "set 7, (/s0+/n0), A", "set 7, (/s0-/n0), A");
		if (!cpu_must_execute()) break;
		cpu_regs()->A = temp8;
		break;

	default:
		cpu_finalize_instruction_decode(instruction, 0);
		cpu_mark_instruction_unknown(instruction);
		break;
	}
}
